<?php
// PACE: PHP Asynchronous Command Engine
// PACE: Version 1.1
//   This class interfaces and provides end user access with the PACE service running on the server.
//   This class is intended to work inside cPanel environments but can be adapted to be used elsewhere.
//   For more information, check out https://www.pace.tech/
//   Last modified 3/21/2022, Kyle Bisignani

class PACE {
	private $cpuser, $inited = FALSE, $execscript, $execargarray= array(), $execport = '4477';
	
	// Constructor to get the logged in user.
	function __construct() {
		$this->cpuser = get_current_user();
	}
	
	// Function called to initialize the class with end user information
	//   Usage: PACE->Init($script, [$arg1, ...]);
	function Init() {
		// If we don't have any args, fail.  We need at least one that contains the path to the script
		if (func_num_args() < 1) {
			trigger_error('You must supply at least one argument, your script path', E_USER_WARNING);
			$this->inited = FALSE;
			return FALSE;
		}
		
		// get all the args into an argument array
		$args = func_get_args();
		
		// Get max argument length
		$sock = @socket_create(AF_INET, SOCK_STREAM, SOL_TCP);
		$maxlen = socket_get_option($sock, SOL_SOCKET, SO_SNDBUF);

		foreach ($args as $arg) {
			// Check to make sure the arguments are all strings
			if (is_array($arg) === TRUE || is_object($arg) === TRUE) {
				trigger_error('Arrays and objects cannot be passed through PACE.  Try serializing and unserializing in your scripts to pass these types of arguments', E_USER_WARNING);
				$this->inited = FALSE;
				return FALSE;
			}
			// And make sure we're all within the max length
			if (count($arg) >= $maxlen) {
				trigger_error('PACE can only handle arguments no greater than '.$maxlen, E_USER_WARNING);
				$this->inited = FALSE;
				return FALSE;
			}
		}
		
		// Shift off the first argument - that's our script to execute
		$this->execscript = array_shift($args);
		
		// Check to make sure we have access to the script.  The follows cPanel server restrictions and prevents someone from running a script from outside their root if enforced
		if (is_file($this->execscript) === FALSE) {
			trigger_error('The script could not be found.  Be sure you provided the correct script name and full, complete path to the script', E_USER_WARNING);
			$this->inited = FALSE;
			return FALSE;
		}
		
		
		// Put the remaining arguments (the commands) into the argument array
		if (count($args) > 0)
			$this->execargarray = $args;
		
		// Set inited to true - lets the Run command know we're all set to go
		$this->inited = TRUE;
		
		return TRUE;
	}
	
	// Function called to process the initialized data
	//   Usage: PACE->Run();
	function Run() {
		// Make sure we're ready to go
		if ($this->inited === FALSE) {
			trigger_error('The PACE call isn\'t ready to run', E_USER_WARNING);
			return FALSE;
		}
		
		// Create a new socket
		$sock = @socket_create(AF_INET, SOCK_STREAM, SOL_TCP);
		if ($sock === FALSE) {
			trigger_error('Socket could not be created', E_USER_WARNING);
			return FALSE;
		}
		if (@socket_bind($sock, '127.0.0.1') === FALSE) {
			trigger_error('Socket could not bind', E_USER_WARNING);
			return FALSE;
		}
		if (@socket_connect($sock, '127.0.0.1', $this->execport) === FALSE) {
			trigger_error('Socket could not connect', E_USER_WARNING);
			return FALSE;
		}

		// Lets write the total length
		$string = $this->cpuser.'|'.$this->execscript."|".implode("|", $this->execargarray);
		$cnt = strlen( $string );
		
		WriteXBytes($sock, PHP_INT_SIZE, int2string($cnt));
		WriteXBytes($sock, $cnt, $string);
		
		// Close
		socket_close($sock);
		
		return TRUE;
	}
}

function WriteXBytes($a_socket, $a_length, $a_data) {
	$bytesSent = 0;
	
	while ($bytesSent < $a_length) {
		$k = socket_write($a_socket, substr($a_data, $bytesSent) );
		if ($k === FALSE) {
			trigger_error('Could not transmit the total length of what we want to send', E_USER_WARNING);
			return FALSE;
		}
		
		if ($k == 0) {
			trigger_error('Premature end of data error', E_USER_WARNING);
			return;
		}
		
		$bytesSent += $k;
	}
}

function int2string($int, $numbytes=4) {
   $str = "";
   for ($i=0; $i < $numbytes; $i++) {
     $str .= chr($int % 256);
     $int = $int / 256;
   }
   return $str;
}